/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCCOMMONDECODERTYPES_H_316980307630119100782140468796647472819
#define VNCCOMMONDECODERTYPES_H_316980307630119100782140468796647472819

/**
 * \file vnccommondecodertypes.h
 */

#include <vnccommontypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \typedef VNCCommonDecoderEventHandle
 * \brief Event handle used by CDF Decoders.
 *
 * A VNCCommonDecoderEventHandle is a platform-specific mechanism for the
 * framework to detect that there has been activity on a CDF Decoder's
 * context.
 *
 * See VNCCommonDecoderGetEventHandles() for a full discussion of how
 * the framework uses VNCCommonDecoderEventHandle.
 *
 * \see VNCCommonDecoderGetEventHandles(), VNCCommonDecoderActivity()
 */
#ifdef _WIN32
    #include <windows.h>
    typedef HANDLE VNCCommonDecoderEventHandle;
#else
    typedef int VNCCommonDecoderEventHandle;
#endif

/**
 * \brief This struct is used to specify details of an event handle.
 */
typedef struct
{
    /**
     * \brief The event handle.
     */
    VNCCommonDecoderEventHandle handle;

    /**
     * \brief If this is equal to vnc_true, the activity callback will be
     * invoked when this event handle is ready for reading from. On Windows,
     * this is always treated as true.
     */
    vnc_bool_t listenForReadReady;

    /**
     * \brief If this is equal to vnc_true, the activity callback will be
     * invoked when this event handle is ready for writing to. On Windows, this
     * is always treated as true.
     */
    vnc_bool_t listenForWriteReady;

} VNCCommonDecoderEventHandleParams;


/**
 * \brief This struct contains information about the SDK making use of the
 * Common Decoder Framework.
 */
typedef struct
{
    /**
     * \brief If this is equal to vnc_true, the SDK supports stream framerate
     * limit requests, using the
     * VNCCommonDecoderSupportingAPIStreamRequestFrameRateLimit() supporting
     * API.
     */
    vnc_bool_t requestStreamFrameRateLimitSupported;

    /**
     * \brief If this is equal to vnc_true, the SDK supports stream IDR frame
     * requests, using the vncCommonDecoderSupportingAPIStreamRequestIDRFrame()
     * supporting API.
     */
    vnc_bool_t requestStreamIDRFrameSupported;

    /**
     * \brief If this is equal to vnc_true, the SDK supports stream reset
     * requests for audio streams, using the
     * vncCommonDecoderSupportingAPIStreamRequestReset() supporting API.
     */
    vnc_bool_t requestStreamResetAudioSupported;

    /**
     * \brief If this is equal to vnc_true, the SDK supports stream reset
     * requests for video streams, using the
     * vncCommonDecoderSupportingAPIStreamRequestReset() supporting API.
     */
    vnc_bool_t requestStreamResetVideoSupported;

    /**
     * \brief If this is equal to vnc_true, the SDK supports stream reset
     * requests for multiplexed audio/video streams, using the
     * vncCommonDecoderSupportingAPIStreamRequestReset() supporting API.
     */
    vnc_bool_t requestStreamResetMultiplexedAudioVideoSupported;

} VNCCommonDecoderSDKAttributes;


/** \defgroup VNCCommonDecoderError VNCCommonDecoderError */

/**
 * \brief Error code type.
 * \ingroup VNCCommonDecoderError
 *
 * These error codes may be returned by decoder callbacks and supporting
 * APIs, and are split into two ranges:
 *
 *  - 0 to (VNCCommonDecoderErrorVENDOR - 1) - these error codes are a mixture of
 *    error codes (e.g.  VNCCommonDecoderErrorNoMemory) and
 *    other error conditions that are pre-defined by RealVNC-provided decoders.
 *    You are encouraged to reuse error codes in this range, rather than
 *    defining your own, as long as it makes sense to do so.
 *  - VNCCommonDecoderErrorVENDOR and above - this range of error codes is reserved
 *    for the use of third parties developing decoders.  It is intended for
 *    error conditions that are specific to particular decoder implementations,
 *    and that do not map closely to the codes in the 0 to
 *    (VNCCommonDecoderErrorVENDOR - 1) range.
 */
typedef vnc_int32_t VNCCommonDecoderError;

/**
 * \brief The operation was successful - no error occurred.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorNone = 0;

/**
 * \brief No memory was available for the requested operation.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorNoMemory = 1;

/**
 * \brief The required resources are unavailable.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorResourceUnavailable = 2;

/**
 * \brief The decoder requires a licensed feature for which the SDK does not
 * have a license.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorFeatureNotLicensed = 3;

/**
 * \brief The value of one of the arguments to the call is not valid.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorInvalidArgument = 4;

/**
 * \brief The framework encountered an unknown internal error.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorFrameworkInternal = 5;

/**
 * \brief There was a problem with the incoming stream data.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorMalformedData = 6;

/**
 * \brief The required resources encountered an error.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorResourceError = 7;

/**
 * \brief The decoder encountered an unknown internal error.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorInternal = 8;

/**
 * \brief The action cannot be performed in the current state.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorIllegalState = 9;

/**
 * \brief The specified media format is not valid.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorInvalidMediaFormat = 10;

/**
 * \brief The requested action cannot be performed on this thread.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorIllegalThread = 11;

/**
 * \brief The provided stream ID is unrecognised. It may have already been
 * destroyed.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorStreamUnknown = 12;

/**
 * \brief The specified stream is of an unexpected type.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorStreamTypeInvalid = 13;

/**
 * \brief Failed to load a required shared library
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorPluginLoadFailed = 14;

/**
 * \brief An offset for third-party decoder-specific error codes.
 * \ingroup VNCCommonDecoderError
 */
static const VNCCommonDecoderError VNCCommonDecoderErrorVENDOR = 0x10000;


/** \defgroup VNCCommonDecoderMediaType VNCCommonDecoderMediaType */

/**
 * \brief General stream type.
 * \ingroup VNCCommonDecoderMediaType
 */
typedef vnc_int32_t VNCCommonDecoderMediaType;

/**
 * \brief A stream containing only audio.
 * \ingroup VNCCommonDecoderMediaType
 */
static const VNCCommonDecoderMediaType VNCCommonDecoderMediaTypeAudio = 0;

/**
 * \brief A stream containing only video.
 * \ingroup VNCCommonDecoderMediaType
 */
static const VNCCommonDecoderMediaType VNCCommonDecoderMediaTypeVideo = 1;

/**
 * \brief A stream containing both audio and video, multiplexed in some sort of
 * transport stream.
 * \ingroup VNCCommonDecoderMediaType
 */
static const VNCCommonDecoderMediaType VNCCommonDecoderMediaTypeMultiplexedAudioVideo = 2;


/** \defgroup VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificType */

/**
 * \brief Specific stream type.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
typedef vnc_int32_t VNCCommonDecoderStreamSpecificType;

/**
 * \brief The stream type is not known.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeUnknown = 0;

/**
 * \brief CarLife video stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarLifeVideo = 1;

/**
 * \brief CarLife media audio stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarLifeAudioMedia = 2;

/**
 * \brief CarLife navigation TTS audio stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarLifeAudioNaviTTS = 3;

/**
 * \brief CarLife voice recognition prompt audio stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarLifeAudioVRTTS = 4;

/**
 * \brief Apple CarPlay video stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarPlayVideo = 5;

/**
 * \brief Apple CarPlay main audio stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarPlayAudioMain = 6;

/**
 * \brief Apple CarPlay alternate audio stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeCarPlayAudioAlternate = 7;

/**
 * \brief WiFi Display primary RTP stream.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeWiFiDisplayPrimary = 8;

/**
 * \brief VA H.264 rectangles from a VNC connection. The VA H.264 headers will
 * be stripped in the provided payloads.
 * \ingroup VNCCommonDecoderStreamSpecificType
 */
static const VNCCommonDecoderStreamSpecificType VNCCommonDecoderStreamSpecificTypeVNCVAH264 = 9;


/** \defgroup VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentType */

/**
 * \brief Stream content type.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
typedef vnc_int32_t VNCCommonDecoderAudioStreamContentType;

/**
 * \brief General media stream.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
static const VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentTypeMedia = 0;

/**
 * \brief Navigation instructions.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
static const VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentTypeNavigation = 1;

/**
 * \brief Voice recognition prompt.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
static const VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentTypeVRPrompt = 2;

/**
 * \brief Phone call audio.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
static const VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentTypePhoneCall = 3;

/**
 * \brief Alerts, such as ringtones and notifications.
 * \ingroup VNCCommonDecoderAudioStreamContentType
 */
static const VNCCommonDecoderAudioStreamContentType VNCCommonDecoderAudioStreamContentTypeAlert = 4;


/** \defgroup VNCCommonDecoderAudioChannelConfig VNCCommonDecoderAudioChannelConfig */

/**
 * \brief Audio channel configuration.
 * \ingroup VNCCommonDecoderAudioChannelConfig
 */
typedef vnc_int32_t VNCCommonDecoderAudioChannelConfig;

/**
 * \brief Audio with one channel.
 * \ingroup VNCCommonDecoderAudioChannelConfig
 */
static const VNCCommonDecoderAudioChannelConfig VNCCommonDecoderAudioChannelConfigMono = 0;

/**
 * \brief Audio with two channels, left and right.
 * \ingroup VNCCommonDecoderAudioChannelConfig
 */
static const VNCCommonDecoderAudioChannelConfig VNCCommonDecoderAudioChannelConfigStereo = 1;


/** \defgroup VNCCommonDecoderAudioSampleFormat VNCCommonDecoderAudioSampleFormat */

/**
 * \brief The audio sample format.
 * \ingroup VNCCommonDecoderAudioSampleFormat
 */
typedef vnc_int32_t VNCCommonDecoderAudioSampleFormat;

/**
 * \brief Each audio sample is an unsigned 8 bit value from 0 to 255.
 * \ingroup VNCCommonDecoderAudioSampleFormat
 */
static const VNCCommonDecoderAudioSampleFormat VNCCommonDecoderAudioSampleFormatU8 = 0;

/**
 * \brief Each audio sample is a signed 16 bit value, in little-endian format.
 * \ingroup VNCCommonDecoderAudioSampleFormat
 */
static const VNCCommonDecoderAudioSampleFormat VNCCommonDecoderAudioSampleFormatS16LE = 1;

/**
 * \brief Each audio sample is a signed 16 bit value, in big-endian format.
 * \ingroup VNCCommonDecoderAudioSampleFormat
 */
static const VNCCommonDecoderAudioSampleFormat VNCCommonDecoderAudioSampleFormatS16BE = 2;


/** \defgroup VNCCommonDecoderH264BitStreamFormat VNCCommonDecoderH264BitStreamFormat */

/**
 * \brief The H.264 bitstream format.
 * \ingroup VNCCommonDecoderH264BitStreamFormat
 */
typedef vnc_int32_t VNCCommonDecoderH264BitStreamFormat;

/**
 * \brief The bitstream format is not specified.
 * \ingroup VNCCommonDecoderH264BitStreamFormat
 */
static const VNCCommonDecoderH264BitStreamFormat VNCCommonDecoderH264BitStreamFormatUnspecified = 0;

/**
 * \brief The Annex B bitstream format.
 * \ingroup VNCCommonDecoderH264BitStreamFormat
 */
static const VNCCommonDecoderH264BitStreamFormat VNCCommonDecoderH264BitStreamFormatAnnexB = 1;

/**
 * \brief The AVCC bitstream format.
 * \ingroup VNCCommonDecoderH264BitStreamFormat
 */
static const VNCCommonDecoderH264BitStreamFormat VNCCommonDecoderH264BitStreamFormatAVCC = 2;


/** \defgroup VNCCommonDecoderH264Profile VNCCommonDecoderH264Profile */

/**
 * \brief The H.264 profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
typedef vnc_int32_t VNCCommonDecoderH264Profile;

/**
 * \brief The profile is not specified.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileUnspecified = 0;

/**
 * \brief Constrained Baseline Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileConstrainedBaseline = 1;

/**
 * \brief Baseline Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileBaseline = 2;

/**
 * \brief Main Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileMain = 3;

/**
 * \brief Extended Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileExtended = 4;

/**
 * \brief Constrained High Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileConstrainedHigh = 5;

/**
 * \brief High Profile.
 * \ingroup VNCCommonDecoderH264Profile
 */
static const VNCCommonDecoderH264Profile VNCCommonDecoderH264ProfileHigh = 6;


/** \defgroup VNCCommonDecoderVideoScanMode VNCCommonDecoderVideoScanMode */

/**
 * \brief Video scan mode.
 * \ingroup VNCCommonDecoderVideoScanMode
 */
typedef vnc_int32_t VNCCommonDecoderVideoScanMode;

/**
 * \brief Progressive scan mode.
 * \ingroup VNCCommonDecoderVideoScanMode
 */
static const VNCCommonDecoderVideoScanMode VNCCommonDecoderVideoScanModeProgressive = 0;

/**
 * \brief Interlaced scan mode.
 * \ingroup VNCCommonDecoderVideoScanMode
 */
static const VNCCommonDecoderVideoScanMode VNCCommonDecoderVideoScanModeInterlaced = 1;


/** \defgroup VNCCommonDecoderStreamPayloadExtraDataType VNCCommonDecoderStreamPayloadExtraDataType */

/**
 * \brief The payload extra data type.
 * \ingroup VNCCommonDecoderStreamPayloadExtraDataType
 */
typedef vnc_int32_t VNCCommonDecoderStreamPayloadExtraDataType;

/**
 * \brief No extra data.
 * \ingroup VNCCommonDecoderStreamPayloadExtraDataType
 */
static const VNCCommonDecoderStreamPayloadExtraDataType VNCCommonDecoderStreamPayloadExtraDataTypeNone = 0;


/** \defgroup VNCCommonDecoderPropertyKey VNCCommonDecoderPropertyKey */

/**
 * \brief Decoder property keys.
 * \ingroup VNCCommonDecoderPropertyKey
 */
typedef vnc_int32_t VNCCommonDecoderPropertyKey;

/**
 * \brief Used to set the visibility of the overlay. The value provided should
 * point to a vnc_bool_t value, set to vnc_true if the overlay should be made
 * visible, and vnc_false if the overlay should be hidden.
 * \ingroup VNCCommonDecoderPropertyKey
 */
static const VNCCommonDecoderPropertyKey VNCCommonDecoderPropertyKeyOverlayVisibility = 1;

/**
 * \brief Used to set the position and size of the overlay. The value provided
 * should point to a VNCRectangle, which will contain the desired screen
 * coordinates where the overlay should be placed.
 * \ingroup VNCCommonDecoderPropertyKey
 */
static const VNCCommonDecoderPropertyKey VNCCommonDecoderPropertyKeyOverlayRectangle = 2;

/**
 * \brief Offset for vendor keys.
 * \ingroup VNCCommonDecoderPropertyKey
 */
static const VNCCommonDecoderPropertyKey VNCCommonDecoderPropertyKeyVENDOR = 0x10000;


/** \defgroup VNCCommonDecoderDataType VNCCommonDecoderDataType */

/**
 * \brief Data type.
 * \ingroup VNCCommonDecoderDataType
 */
typedef vnc_int32_t VNCCommonDecoderDataType;

/**
 * \brief vnc_bool_t
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeBoolean = 1;

/**
 * \brief vnc_int32_t
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeS32 = 2;

/**
 * \brief vnc_uint32_t
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeU32 = 3;

/**
 * \brief vnc_int64_t
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeS64 = 4;

/**
 * \brief vnc_uint64_t
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeU64 = 5;

/**
 * \brief VNCRectangle
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeRectangle = 6;

/**
 * \brief const char*
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeString = 7;

/**
 * \brief Offset for vendor types.
 * \ingroup VNCCommonDecoderDataType
 */
static const VNCCommonDecoderDataType VNCCommonDecoderDataTypeVENDOR = 0x10000;


/**
 * \brief This struct is used to provide details of a payload.
 */
typedef struct {

  /**
   * \brief A pointer to the stream payload data. This will vary depending on the
   * media type. The lifetime and ownership of this data is tied to the payload
   * struct.
   */
  vnc_uint8_t* payloadData;

  /**
   * \brief The size of the payload data pointed to by payloadData.
   */
  vnc_size_t payloadDataLength;

  /**
   * \brief Set to vnc_true if and only if payloadReceiveTimestampMs is known and
   * set to the correct value.
   */
  vnc_bool_t hasPayloadReceiveTimestamp;

  /**
   * \brief The time at which the payload was received, in milliseconds. This
   * should be treated as a relative timestamp rather than an absolute one.
   * Should be ignored if hasPayloadReceiveTimestamp is false.
   */
  vnc_uint64_t payloadReceiveTimestampMs;

  /**
   * \brief Set to vnc_true if and only if payloadPresentationTimestampMs is
   * known and set to the correct value.
   */
  vnc_bool_t hasPayloadPresentationTimestamp;

  /**
   * \brief The time at which the payload should be rendered, in milliseconds.
   * This should be treated as a relative timestamp rather than an absolute one.
   * Should be ignored if hasPayloadPresentationTimestamp is false.
   */
  vnc_uint64_t payloadPresentationTimestampMs;

  /**
   * \brief See extraData below.
   */
  VNCCommonDecoderStreamPayloadExtraDataType extraDataType;

  /**
   * \brief Some payload types, e.g. M2TS over RTP, may offer additional data
   * about each payload, for example sequence numbers. If so, extraData will
   * point to a struct containing this data, and extraDataType will let the
   * decoder know the type of struct to cast it to.
   */
  void* extraData;

  /**
   * \brief A context pointer owned by the SDK, for the purposes of identifying
   * and managing this payload object. Should not be accessed by the decoder.
   */
  void* privateData;

} VNCCommonDecoderStreamPayload;


/**
 * \brief This struct is used to provide details of an LPCM stream. All
 * elements must be filled in.
 *
 * The contents of this struct will not change in future versions, so a size
 * parameter is not required for APIs using it.
 */
typedef struct {

  /**
   * \brief The number of samples per second. This is not affected by the channel
   * count.
   */
  vnc_uint32_t sampleRateHz;

  /**
   * \brief The channel configuration of this stream.
   */
  VNCCommonDecoderAudioChannelConfig channelConfig;

  /**
   * \brief The bit layout of each sample.
   */
  VNCCommonDecoderAudioSampleFormat sampleFormat;

} VNCCommonDecoderAudioFormatDetailsLPCM;


/**
 * \brief This struct is used to provide details of an H.264 stream. All
 * elements must be filled in.
 */
typedef struct {

  /**
   * \brief The H.264 bitstream format (Annex B, AVCC, etc)
   */
  VNCCommonDecoderH264BitStreamFormat bitStreamFormat;

  /**
   * \brief The H.264 profile.
   */
  VNCCommonDecoderH264Profile profile;

  /**
   * \brief The major part of the H.264 level. If this is unknown, the value -1
   * should be used.
   */
  vnc_int32_t levelMajor;

  /**
   * \brief The minor part of the H.264 level. If this is unknown, the value -1
   * should be used.
   */
  vnc_int32_t levelMinor;

} VNCCommonDecoderH264Attributes;


/**
 * \brief This struct is used to provide details of a video stream. All
 * elements must be filled in.
 *
 * The contents of this struct will not change in future versions, so a size
 * parameter is not required for APIs using it.
 */
typedef struct {

  /**
   * \brief The video width, in pixels.
   */
  vnc_uint32_t width;

  /**
   * \brief The video height, in pixels.
   */
  vnc_uint32_t height;

  /**
   * \brief The framerate, in frames per second.
   */
  vnc_uint32_t framesPerSecond;

  /**
   * \brief The type of scan mode used (interlaced or progressive).
   */
  VNCCommonDecoderVideoScanMode scanMode;

} VNCCommonDecoderVideoMode;




/**
 * \brief Represents a loaded decoder plugin.
 *
 * This type is opaque to the SDK, which will pass the value returned by
 * VNCCommonDecoderFactoryInitialize() to any further decoder callbacks that
 * may require it.
 */
typedef struct VNCCommonDecoderPluginContextImpl* VNCCommonDecoderPluginContext;

/**
 * \brief Identifies the internal data structures of the SDK that correspond to
 * a loaded decoder plugin.
 *
 * This type is opaque to the decoder implementation, which must pass
 * the value supplied to VNCCommonDecoderFactoryInitialize() to any
 * supporting API calls that may require it.
 */
typedef struct VNCCommonDecoderFrameworkContextImpl* VNCCommonDecoderFrameworkContext;


/**
 * \brief An arbitrary pointer which can be passed unmodified from the viewer
 * application to the decoder, through the SDK.
 */
typedef struct VNCCommonDecoderApplicationContextImpl* VNCCommonDecoderApplicationContext;



/**
 * \brief A structure describing the decoder.
 */
typedef struct
{
  /**
   * \brief The human-readable name of the decoder, encoded as modified UTF-8
   * with a NUL terminator. This will be used in logging/debug messages.
   */
  char* decoderName;

  /**
   * \brief The human-readable version of the decoder, encoded as modified
   * UTF-8 with a NUL terminator. This will be used in logging/debug messages.
   */
  char* decoderVersion;

} VNCCommonDecoderSupportInfo;


/**
 * \brief Parameters which can be set by the decoder on creation.
 */
typedef struct
{
  /**
   * \brief Some SDKs may attempt to buffer payloads before passing them to the
   * decoder. If this flag is enabled, the SDK should disable such buffering.
   */
  vnc_bool_t disableBuffering;

} VNCCommonDecoderCreationParams;

#ifdef __cplusplus
}
#endif

#endif // VNCCOMMONDECODERTYPES_H_316980307630119100782140468796647472819

